/* 
 * SearchPanel.java
 * 
 * This Class creates a custom panel for the repository and album tab. Allow user
 * to search via the search functions and parameters.
 * 
 */ 

//import statements
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import javax.swing.border.*;
import javax.swing.plaf.basic.BasicFileChooserUI;
import java.util.*;

/* SearchPanel class
 * 
 * The purpose of this class is to create a custom panel for the album and
 * repository tab so search can be performed on both panel.
 * 
 * Written by : Roger Yang
 *                                                                          
 * 
 * Version 1.4 29 Nov 2006
 * 
  * Revision    Status       Date            By: 
 * 1.0         Created      6 Nov 2006     Roger Yang       
 * 1.1		   Modified    12 Nov 2006	   Roger Yang                                                              
 * 1.2		   Modified    23 Nov 2006	   Roger Yang                                                       
 * 1.3		   Modified    24 Nov 2006	   Roger Yang 
 * 1.4		   Modified    29 Nov 2006	   Roger Yang 
 */  
public class SearchPanel extends JPanel
{
	//Create a DataStructure for storage
	DescriptorDataStructure DescriptorData = new DescriptorDataStructure();
	
	//declare vectors for dynamically add radio button and textfield
	Vector<JTextField> tf = new Vector<JTextField>();
	Vector<JRadioButton> rb = new Vector<JRadioButton>();

	//Declare all subpanels for the main scrollpane
	JPanel pSearchSelect = new JPanel();
	JPanel pSearchButtons = new JPanel();
	JPanel pSearchOption = new JPanel();
	
	//Declare buttons for the search panel
	JButton SearchOK = new JButton("OK");
	JButton SearchClear = new JButton("Clear");
	
	//Declare radio buttons for the search panel	
	JRadioButton jrbAnd = new JRadioButton("AND");
	JRadioButton jrbOr = new JRadioButton("OR");
	JRadioButton jrbAll = new JRadioButton("ALL");
	
	//Declare the main scrollpane
	JScrollPane pSelectScroll = new JScrollPane();
	//declare variable for scrollheight calcuation
	int ScrollHeight;
	
	//create Action handler
	ActionHandler e = new ActionHandler();
	
	/** 
     * Class Constructor
     * This function sets the default values for the gui and variables
     * when the class is first initated.
     */ 
	public SearchPanel()
	{
		//set the title for each sub panels
		pSearchOption.setBorder(new TitledBorder("Search Option"));
		pSearchSelect.setBorder(new TitledBorder("Search Detail"));
	
		//set the layout for each subpanels
		pSearchOption.setLayout(new GridLayout(3,1));
		pSearchSelect.setLayout(new GridBagLayout());
		pSearchButtons.setLayout(new GridLayout(2,1));
		
		//set the layout for the main panels
		this.setLayout(new BorderLayout());
		
		//set the preferred size for the descriptor type subpanel
		pSearchSelect.setPreferredSize(new Dimension(200,100));
		
		//set the default selected radiobutton
		jrbAll.setSelected(true);
		
		//add radio button to the otion panel
		pSearchOption.add(jrbAnd);
		pSearchOption.add(jrbOr);
		pSearchOption.add(jrbAll);
		
		//group radio buttons into one group so only 1 can be selected at each time
		ButtonGroup bg1 = new ButtonGroup();
		bg1.add(jrbAnd);
		bg1.add(jrbOr);
		bg1.add(jrbAll);
		
		
		
		//add ActionListener
		SearchOK.addActionListener(e);
		SearchClear.addActionListener(e);
		jrbAnd.addActionListener(e);
		jrbOr.addActionListener(e);
		
        //set default value for the height of the panel inside the scrollpanel
        ScrollHeight = 0;
		for(int i = 0; i < DescriptorData.size(); i++)
        {
            //for each descriptor type found in database, make the panel larger
            ScrollHeight += 50;
            //add radio button for each descriptor type to track
            rb.add(i,new JRadioButton(DescriptorData.GetDescriptorType(i)));
            //add textfield for each descriptory type to track
            tf.add(i,new JTextField());
            //add the radio button and textfield to the panel
            addComponents(rb.get(i), tf.get(i), pSearchSelect);
        }
		
		//add buttons to the subpanel
		pSearchButtons.add(SearchOK);
		pSearchButtons.add(SearchClear);
		
		//set preferred size of the panel inside the scroll panel
		pSearchSelect.setPreferredSize(new Dimension(200,ScrollHeight));
		//add the inside panel to the scroll panel
		pSelectScroll.getViewport().add(pSearchSelect);
		//Add sub-Panel for the search panel
		this.setPreferredSize(new Dimension(250,500));
		
		//add each subpanel to the main panel
		this.add(pSearchOption,BorderLayout.NORTH);
		this.add(pSelectScroll,BorderLayout.CENTER);
		this.add(pSearchButtons,BorderLayout.SOUTH);
	}
	
	/** 
     * updateSelectPanel
     *
     * This function update the search panel when needed.  For example, if 
     * a new descriptor type is added to the repository or when the textboxes
     * needs to be cleared.
     *
     * Precondition : no precondition
     * Postcondition : updates the search panel with new descriptors and clear 
     *					all textboxes
     */ 
	public boolean updateSelectPanel()
	{
        
		//remove everything on the subpanel before reload it
		pSearchSelect.removeAll();
		
		//remove all elements in the data structure to reload it
		rb.removeAllElements();
		tf.removeAllElements();
		
		//reset the height of the panel to 0
		ScrollHeight = 0;
		
		
		for(int i = 0; i < DescriptorData.size(); i++)
        {
            //for each descriptor type found in database, make the panel larger
            ScrollHeight += 50;
            //add radio button for each descriptor type to track
            rb.add(i,new JRadioButton(DescriptorData.GetDescriptorType(i)));
            //add textfield for each descriptory type to track
            tf.add(i,new JTextField());
            //add the radio button and textfield to the panel
            addComponents(rb.get(i), tf.get(i), pSearchSelect);
        }
        //set preferred size
        pSearchSelect.setPreferredSize(new Dimension(200,ScrollHeight));
        //update the panel
        pSearchSelect.updateUI();
   		//return true so if outside class need to know if it was successful
		return true;
	}
	
	/** 
     * Search
     *
     * This function triggers the search by using the search functions and returns
     * a vector which contains imageIDs as the search result
     *
     * Precondition : no precondition
     * Postcondition : returns the search result to calling class
     */ 
	public Vector<Integer> Search()
	{
		//create a class for search function
		SearchFunction sf = new SearchFunction();
		
		/**
		 *create 2 vectors to store which descriptor is selected and what
		 *value are we searching for
		 */
		Vector<Integer> SearchNum = new Vector<Integer>();
		Vector<String> SearchVal = new Vector<String>();
		
		//create a vector to store the search result from the search functions
		Vector<Integer> SearchResult = new Vector<Integer>();
		//set default search parameter to all
		int SearchParam = 1;
		
		//if and radio button is selected then set param to AND
		if (jrbAnd.isSelected())
		{
			SearchParam = 2;
		}
		//if or radio button is selected then set param to OR
		else if (jrbOr.isSelected())
		{
			SearchParam = 3;
		}
		//else set param to ALL
		else 
		{
			SearchParam = 1;
		}
		
		//for each radio button on the search page
		for (int i=0; i<rb.size();i++)
		{
			//if the radio button is selected
			if (rb.get(i).isSelected())
			{
				//add this descriptor type to the vector
				SearchNum.add(new Integer(i));
				//add the value next to this descriptor type to vector
				SearchVal.add(tf.get(i).getText());
			}
		}
		
		/*
		 *if no radio button is selected and parameter is not ALL
		 *alert user that atleast 1 radio button must be selected
		 */
		if (SearchNum.size() ==0 && SearchParam!=1)
		{
			JOptionPane.showMessageDialog(null,
					"Atleaset 1 Search Parameter must be selected");
		}
		
		//pass the search parameters to the search function and get the results
		SearchResult = sf.search(SearchParam,DescriptorData.GetVector(),SearchNum, SearchVal);
		
		//return the result to the calling class
		return SearchResult;
	}
	
	/** 
     * Class Actionhandler
     * This function handles all the actions from the actionlisteners
     * NOTE: that the ok button is not catched from this class
     *		 any class uses this search panel will have to listen to the 
     *		 ok button.
     */ 
	public class ActionHandler implements ActionListener
	{
		
		public void actionPerformed(ActionEvent e)
		{
			//if clear button is clicked
			if (e.getSource() == SearchClear)
			{
				//update and thus clear the search panel
				updateSelectPanel();
			}
			//calling class listen to the ok button
			/*if (e.getSource() == SearchOK)
			{
				
			}*/
			
		}
	}
	
	//Helper Functions==========================================================
	
		/** 
     * addComponents
     * This function takes a radio button, a textfield and add it to the panel 
     * of a gridbaglayout
     * Precondition :  radiobutton and textfield are set, and the panel is of a 
     *					gridbaglayout.
     * Postcondition : the component is added and the spacing is according to
     *				   the gridbag constraints
     */
	private static void addComponents(JRadioButton RadioButton,JTextField tf,
                                      JPanel panel)
    {
        //create a gribbag contraint object
        GridBagConstraints gbc = new GridBagConstraints();
        //set the distance between each element
        gbc.insets = new Insets(2,2,2,2);
        //set the weight and thus the space of that the element takes
        gbc.weighty = 1.0;
        //set the width of the element
        gbc.gridwidth = gbc.RELATIVE;
        //set the weight and thus the space of that the element takes
        gbc.weightx = 0.1;
        //set contraints
        gbc.fill = gbc.NONE;
        //set alignment
        gbc.anchor = gbc.WEST;
        //add radiobutton
        panel.add(RadioButton, gbc);
        //set size of the textfield
        gbc.gridwidth = gbc.REMAINDER;
        //set contstraints
        gbc.weightx = 1.0;
        gbc.fill = gbc.HORIZONTAL;
        //set alignment
        gbc.anchor = gbc.WEST;
        //add textfield
        panel.add(tf, gbc);
    }
    //End Helper Functions======================================================
}